/* mfwindow.c -- Presentation Manager window for METAFONT.

   Copyright (C) 1994 Ralph Schleicher  */

/* This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */


#define USE_OS2_TOOLKIT_HEADERS		/* In IBM I trust. */

#define INCL_PM
#define INCL_DOS

#include <os2.h>
#include <getopt.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <mftalk.h>
#include "resources.h"


/* Local definitions. */

#define achAppName	"METAFONT Window"
#define achClientClass	"METAFONT"

#define DOS		1		/* Used with Bye(). */
#define WIN		2
#define LINE(back)	(__LINE__ - (back))

/* Foreward declarations.  */

VOID Main (VOID);
VOID Talk (ULONG *ulArgument);
VOID Exit (ULONG ulExitCode);
VOID Bye (ULONG ulMode, ...);

MRESULT ClientProc (HWND hwnd, ULONG ulMessage, MPARAM mp1, MPARAM mp2);
MRESULT Create (HWND hwnd);
MRESULT Resize (HWND hwnd, MPARAM mp1, MPARAM mp2);
MRESULT Paint (HWND hwnd);
MRESULT HorizScroll (HWND hwnd, MPARAM mp1, MPARAM mp2);
MRESULT VertScroll (HWND hwnd, MPARAM mp1, MPARAM mp2);
MRESULT KeyEvent (HWND hwnd, MPARAM mp1, MPARAM mp2);
MRESULT Command (HWND hwnd, MPARAM mp1, MPARAM mp2);

LONG RoundDown (LONG lArg, LONG lMod);
LONG RoundUp (LONG lArg, LONG lMod);

/* Constant values. */

const ULONG ulPageSize = 4096;		/* Page size of the i386. */

/* Variables storing command line arguments.  */

ULONG ulWidth = 1664;			/* Width of METAFONT's window. */
ULONG ulHeight = 1200;			/* Height of METAFONT's window. */
HFILE hfInput = -1;			/* File handle for listening. */
HFILE hfOutput = -1;			/* File handle for speaking. */
PID pidMetafont = 0;			/* METAFONT's process ID. */

/* Main and client window handles.  */

HAB habMain = 0;			/* Anchor block handle. */
HMQ hmqMain = 0;			/* Message queue handle. */
HWND hwndMain = 0;			/* Frame window handle. */

HWND hwndClient = 0;			/* Client window handle. */
HWND hwndMenu = 0;			/* Menu bar handle. */
HWND hwndHorizScroll = 0;		/* Horizontal scroll bar handle. */
HWND hwndVertScroll = 0;		/* Vertical scroll bar handle. */

HSWITCH hswitchMetafont = 0;		/* Window list handle of METAFONT. */

/* Identifiers of the thread talking with METAFONT.  */

TID tidTalk = 0;			/* The thread ID. */
HEV hevTalk = 0;			/* Posted if initialized. */
HAB habTalk = 0;			/* Anchor block handle. */

/* Handles and other variables for painting.  */

HDC hdcScreen = 0;			/* The device context of the screen. */
HDC hdcMemory = 0;			/* One for inside core. */
HDC hdcPrinter = 0;			/* Another one for hardcopy proofs. */

HPS hpsScreen = 0;			/* The visible presentation space. */
HPS hpsMemory = 0;			/* Presentation space for bitmaps. */

HBITMAP hbmImage = 0;			/* An image of METAFONT's window. */
HMTX hmtxImage = 0;			/* Serialize access to it. */

SIZEL sizlGeometry;			/* Geometry of the client window. */
LONG alColorMap[2];			/* METAFONT's color map. */

/* Some extra information for scrolling the contents of the client window.  */

POINTS ptsByteAlign;			/* Speed up scrolling. */

POINTS ptsScrollPos;			/* Current scroll bar position. */
POINTS ptsScrollOld;			/* Old scroll bar position. */
POINTS ptsScrollSize;			/* Size of the client window. */
POINTS ptsScrollPage;			/* Amount when scrolling a page. */
POINTS ptsScrollLine;			/* Amount when scrolling a line */

/* Global declarations for API/GPI return values.  */

APIRET apiretError = 0;
BOOL fSuccess = TRUE;


/* METAFONT puts the origin of its coordinate system (but only for online
   displays) into the upper left corner but the Presentation Manager uses
   the lower left corner by default (what a kludge).  This transformation
   will be applied during input for the sake of speed.  */

static inline LONG
InputX (LONG lX)
{
  return lX;
}

static inline LONG
InputY (LONG lY)
{
  return sizlGeometry.cy - lY;
}


/* The entry point of a PM application is still main().  Do everything we
   have to do with pure C, then switch to the OS/2 API.  */

int
main (int argc, char **argv)
{
  int c, w, h, i, o, p;

  w = (int) ulWidth;
  h = (int) ulHeight;
  i = (int) hfInput;
  o = (int) hfOutput;
  p = (int) pidMetafont;

  opterr = 0;

  while (1)
    {
      c = getopt (argc, argv, "w:h:i:o:p:");
      if (c == EOF)
	break;
      switch (c)
	{
	case 'w':
	  if (sscanf (optarg, "%d", &w) != 1)
	    return 1;
	  break;
	case 'h':
	  if (sscanf (optarg, "%d", &h) != 1)
	    return 1;
	  break;
	case 'i':
	  if (sscanf (optarg, "%d", &i) != 1)
	    return 1;
	  break;
	case 'o':
	  if (sscanf (optarg, "%d", &o) != 1)
	    return 1;
	  break;
	case 'p':
	  if (sscanf (optarg, "%d", &p) != 1)
	    return 1;
	  break;
	default:
	  return 1;
	}
    }

  if (w < 8 || h < 8 || i < 0 || o < 0  || p < 1)
    return 1;

  ulWidth = (ULONG) w;
  ulHeight = (ULONG) h;
  hfInput = (HFILE) i;
  hfOutput = (HFILE) o;
  pidMetafont = (PID) p;

  Main ();

  return 0;
}


/* The usual body of a PM application.  */

VOID
Main (VOID)
{
  QMSG qmsgMessage;			/* Message structure. */
  ULONG ulFlags;			/* Frame window flags. */

  /* Install an exit procedure cleaning things up.  */

  apiretError = DosExitList (EXLST_ADD, Exit);
  if (apiretError)
    DosExit (EXIT_PROCESS, 1);

  /* Get access to the window system.  */

  habMain = WinInitialize (0);
  if (!habMain)
    Bye (WIN, 0, "WinInitialize", LINE (2));

  hmqMain = WinCreateMsgQueue (habMain, 0);
  if (!hmqMain)
    Bye (WIN, habMain, "WinCreateMsgQueue", LINE (2));

  /* Create the main frame window and its clients.  */

  fSuccess = WinRegisterClass (habMain,
    achClientClass, ClientProc, CS_SIZEREDRAW, 0);
  if (!fSuccess)
    Bye (WIN, habMain, "WinRegisterClass", LINE (3));

  ulFlags = FCF_SYSMENU | FCF_TITLEBAR | FCF_SIZEBORDER | FCF_MINMAX |
    FCF_HORZSCROLL | FCF_VERTSCROLL | FCF_SHELLPOSITION | FCF_TASKLIST |
    FCF_MENU | FCF_ICON | FCF_ACCELTABLE;
  hwndMain = WinCreateStdWindow (HWND_DESKTOP, WS_VISIBLE, &ulFlags,
    achClientClass, achAppName, WS_VISIBLE, 0, ID_RESOURCES, &hwndClient);
  if (!hwndMain)
    Bye (WIN, habMain, "WinCreateStdWindow", LINE (3));

  /* Graphics is present.  Start the thread talking with METAFONT.  */

  apiretError = DosCreateEventSem (NULL, &hevTalk, 0, FALSE);
  if (apiretError)
    Bye (DOS, "DosCreateEventSem", LINE (2));

  apiretError = DosCreateThread (&tidTalk, (PFNTHREAD) Talk, 0, 0, 4096);
  if (apiretError)
    Bye (DOS, "DosCreateThread", LINE (2));

  apiretError = DosWaitEventSem (hevTalk, SEM_INDEFINITE_WAIT);
  if (apiretError)
    Bye (DOS, "DosWaitEventSem", LINE (2));

  apiretError = DosCloseEventSem (hevTalk);
  if (apiretError)
    Bye (DOS, "DosCloseEventSem", LINE (2));

  hevTalk = 0;

  /* Process all messages in the message queue and exit if this is done.  */

  while (WinGetMsg (habMain, &qmsgMessage, 0, 0, 0))
    WinDispatchMsg (habMain, &qmsgMessage);

  DosExit (EXIT_PROCESS, 0);
}


/* This is the thread talking with METAFONT (the argument is not used).  */

VOID
Talk (ULONG *ulArgument)
{
  ULONG ulAccess;			/* Access mode of a stream. */
  LONG lCode;				/* Graphics command. */
  LONG *plTalk;				/* Data buffer for talking. */
  LONG *plColumn;			/* Pointer to "plTalk". */
  ULONG ulAlloc;			/* Size of "plTalk". */
  ULONG ulSize;				/* Bytes to read. */
  ULONG ulBytes;			/* Bytes read. */
  ULONG ulCount;			/* Counter for "plColumn". */
  LONG lIndex;				/* Color index into "alColorMap". */
  LONG lColor;				/* Actual color value. */
  POINTL ptlPoint;			/* Point in the presentation space. */

  /* Check if we can have a conversation with METAFONT.  */
   
  apiretError = DosQueryFHState (hfInput, &ulAccess);
  if (apiretError || (ulAccess & 0x07) != OPEN_ACCESS_READONLY)
    Bye (DOS, "DosQueryFHState", LINE (2));

  apiretError = DosQueryFHState (hfOutput, &ulAccess);
  if (apiretError || (ulAccess & 0x07) != OPEN_ACCESS_WRITEONLY)
    Bye (DOS, "DosQueryFHState", LINE (2));

  /* Send METAFONT our acknowledgment.  We can close the output stream after
     this action. */

  lCode = MF_ACK;

  apiretError = DosWrite (hfOutput, &lCode, sizeof (LONG), &ulBytes);
  if (apiretError)
    Bye (DOS, "DosWrite", LINE (2));

  apiretError = DosClose (hfOutput);
  if (apiretError)
    Bye (DOS, "DosClose", LINE (2));

  /* Get some memory for the unknown number of columns coming along with the
     line-command. */

  ulAlloc = ulPageSize;

  apiretError = DosAllocMem ((VOID **) &plTalk,
    ulAlloc, PAG_COMMIT | PAG_READ | PAG_WRITE);
  if (apiretError)
    Bye (DOS, "DosAllocMem", LINE (3));

  /* Gain access to the window system and say that we start talking.  */

  habTalk = WinInitialize (0);
  if (!habTalk)
    DosExit (EXIT_PROCESS, 1);

  apiretError = DosPostEventSem (hevTalk);
  if (apiretError)
    Bye (DOS, "DosPostEventSem", LINE (2));

  /* Create a graphics segment.  */

  fSuccess = GpiOpenSegment (hpsMemory, 1);
  if (!fSuccess)
    Bye (WIN, habTalk, "GpiOpenSegment", LINE (2));

  while (1)
    {
      /* Read the next command code from METAFONT.  */

      apiretError = DosRead (hfInput, &lCode, sizeof (LONG), &ulBytes);
      if (apiretError)
	Bye (DOS, "DosRead", LINE (2));

      switch (lCode)
	{
	case MF_LINE:

	  /* Read the color index, the start point and the number of
             remaining columns.  */

	  apiretError = DosRead (hfInput, plTalk, 4 * sizeof (LONG), &ulBytes);
	  if (apiretError)
	    Bye (DOS, "DosRead", LINE (2));

	  /* The color index of the initial line comes first.  */

	  lIndex = plTalk[0];
	  lColor = alColorMap[lIndex];

	  fSuccess = GpiSetColor (hpsMemory, lColor);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiSetColor", LINE (2));

	  /* The next two numbers are the coordinates of the start point.  */

	  ptlPoint.x = InputX (plTalk[1]);
	  ptlPoint.y = InputY (plTalk[2]);

	  fSuccess = GpiMove (hpsMemory, &ptlPoint);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiMove", LINE (2));

	  /* The fourth number is the number of remaining columns.  We
	     increase the buffer if there is not enough memory.  The stuff
	     inside the if-statement is IBM's idea of realloc().  */

	  ulCount = plTalk[3];
	  ulSize = ulCount * sizeof (LONG);

	  if (ulSize > ulAlloc)
	    {
	      ULONG ulMore, ulMask;

	      ulMask = ulPageSize - 1;
	      ulMore = (ulSize - ulAlloc + ulMask) & ~ulMask;

	      apiretError = DosSetMem ((VOID *) (plTalk + ulAlloc),
	        ulMore, PAG_DEFAULT);
	      if (apiretError)
		Bye (DOS, "DosSetMem", LINE (3));

	      ulAlloc += ulMore;
	    }

	  apiretError = DosRead (hfInput, plTalk, ulSize, &ulBytes);
	  if (apiretError)
	    Bye (DOS, "DosRead", LINE (2));

	  /* Draw the sequence of horizontal lines.  */

	  for (plColumn = plTalk; ulCount; --ulCount)
	    {
	      /* Fix the end point of the line and draw it.  */

	      ptlPoint.x = InputX (*plColumn); ++plColumn;

	      fSuccess = GpiLine (hpsMemory, &ptlPoint);
	      if (!fSuccess)
		Bye (WIN, habTalk, "GpiLine", LINE (2));

	      /* Invert the drawing color and continue.  */

	      lIndex = 1 - lIndex;
	      lColor = alColorMap[lIndex];

	      fSuccess = GpiSetColor (hpsMemory, lColor);
	      if (!fSuccess)
		Bye (WIN, habTalk, "GpiSetColor", LINE (2));
	    }

	  break;

	case MF_FLUSH:

	  /* Finish the current graphics segment and paint it into the
	     bitmap.  */

	  fSuccess = GpiCloseSegment (hpsMemory);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiCloseSegment", LINE (2));

	  apiretError = DosRequestMutexSem (hmtxImage, SEM_INDEFINITE_WAIT);
	  if (apiretError)
	    Bye (DOS, "DosRequestMutexSem", LINE (2));

	  fSuccess = GpiDrawSegment (hpsMemory, 1);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiDrawSegment", LINE (2));

	  apiretError = DosReleaseMutexSem (hmtxImage);
	  if (apiretError)
	    Bye (DOS, "DosReleaseMutexSem", LINE (2));

	  /* Put the main frame window into the foreground (but don't give
	     it the keyboard focus) and redraw the whole client window.  */

	  fSuccess = WinSetWindowPos (hwndMain,
            HWND_TOP, 0, 0, 0, 0, SWP_ZORDER);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "WinSetWindowPos", LINE (3));

	  fSuccess = WinInvalidateRect (hwndClient, NULL, FALSE);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "WinInvalidateRect", LINE (2));

	  /* Create the next segment ...  */

	  fSuccess = GpiDeleteSegment (hpsMemory, 1);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiDeleteSegment", LINE (2));

	  fSuccess = GpiOpenSegment (hpsMemory, 1);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiOpenSegment", LINE (2));

	  break;

	case MF_RECT:

	  /* Read color index and corner points.  */

	  apiretError = DosRead (hfInput, plTalk, 5 * sizeof (LONG), &ulBytes);
	  if (apiretError)
	    Bye (DOS, "DosRead", LINE (2));

	  /* Set the drawing color.  */

	  fSuccess = GpiSetColor (hpsMemory, alColorMap[plTalk[0]]);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiSetColor", LINE (2));

	  /* Go to the lower left corner of the rectangle.  */

	  ptlPoint.x = InputX (plTalk[1]);
	  ptlPoint.y = InputY (plTalk[2]);

	  fSuccess = GpiMove (hpsMemory, &ptlPoint);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiMove", LINE (2));

	  /* Draw a filled rectangle.  */

	  ptlPoint.x = InputX (plTalk[3]);
	  ptlPoint.y = InputY (plTalk[4]);

	  fSuccess = GpiBox (hpsMemory, DRO_OUTLINEFILL, &ptlPoint, 0, 0);
	  if (!fSuccess)
	    Bye (WIN, habTalk, "GpiBox", LINE (2));

	  break;

	default:

	  /* This shouldn't happen. */

	  Bye (0, "Talk: Unknow opcode %#x", lCode);
	}
    }
}


/* Here is the exit procedure installed with Main().  This simplifies error
   handling enormously.  */

VOID
Exit (ULONG ulExitCode)
{
  /* Terminate the asynchronous thread.  */

  if (tidTalk)
    {
      DosSuspendThread (tidTalk);

      if (habTalk)
	WinTerminate (habTalk);

      DosKillThread (tidTalk);
    }

  if (hevTalk)
    DosCloseEventSem (hevTalk);

  /* The presentation spaces should be destroyed explicitly.  */

  if (hmtxImage)
    DosCloseMutexSem (hmtxImage);
  if (hbmImage)
    GpiDeleteBitmap (hbmImage);

  if (hpsMemory)
    {
      GpiAssociate (hpsMemory, NULLHANDLE);
      GpiDestroyPS (hpsMemory);
    }

  if (hpsScreen)
    {
      GpiAssociate (hpsScreen, NULLHANDLE);
      GpiDestroyPS (hpsScreen);
    }

  /* Destroy the main frame window. */

  if (habMain)
    {
      if (WinIsWindow (habMain, hwndMain))
	WinDestroyWindow (hwndMain);

      if (hmqMain)
	WinDestroyMsgQueue (hmqMain);

      WinTerminate (habMain);
    }

  DosExitList (EXLST_EXIT, Exit);
}


/* It's nice to have some information if an error occurs.  */

VOID
Bye (ULONG ulMode, ...)
{
  CHAR achBuffer[512];			/* Buffer for the error message. */
  HAB habThread;			/* Find the correct window error. */
  CHAR *pchName;			/* Name of the failling function. */
  ULONG ulLine;				/* Line number where is happend. */
  CHAR *pchFormat;			/* Free format string. */
  ULONG ulResponse;			/* Response from the message box. */
  va_list vaPointer;			/* A good old C pointer. */

  va_start (vaPointer, ulMode);

  switch (ulMode)
    {
    case DOS:

      pchName = va_arg (vaPointer, CHAR *);
      ulLine = va_arg (vaPointer, ULONG);

      sprintf (achBuffer, "%s:%u: Return code = %d",
	pchName, ulLine, apiretError);

      break;

    case WIN:

      habThread = va_arg (vaPointer, HAB);
      pchName = va_arg (vaPointer, CHAR *);
      ulLine = va_arg (vaPointer, ULONG);

      if (habThread)
	sprintf (achBuffer, "%s:%u: Last error = %#x",
	  pchName, ulLine, WinGetLastError (habThread));
      else
	sprintf (achBuffer, "%s:%u: Fatal error", pchName, ulLine);

      break;

    default:

      pchFormat = va_arg (vaPointer, CHAR *);

      vsprintf (achBuffer, pchFormat, vaPointer);
    }

  va_end (vaPointer);

  ulResponse = WinMessageBox (HWND_DESKTOP, HWND_DESKTOP, achBuffer,
    achAppName, 0, MB_OKCANCEL | MB_WARNING | MB_MOVEABLE);
  if (ulResponse == MBID_CANCEL)
    DosExit (EXIT_PROCESS, 1);
}


/* Catch the messages we have to look for.  */

MRESULT
ClientProc (HWND hwnd, ULONG ulMessage, MPARAM mp1, MPARAM mp2)
{
  switch (ulMessage)
    {
    case WM_PAINT:
      return Paint (hwnd);
    case WM_HSCROLL:
      return HorizScroll (hwnd, mp1, mp2);
    case WM_VSCROLL:
      return VertScroll (hwnd, mp1, mp2);
    case WM_CHAR:
      return KeyEvent (hwnd, mp1, mp2);
    case WM_SIZE:
      return Resize (hwnd, mp1, mp2);
    case WM_COMMAND:
      return Command (hwnd, mp1, mp2);
    case WM_CREATE:
      return Create (hwnd);
    }

  return WinDefWindowProc (hwnd, ulMessage, mp1, mp2);
}


/* Setup the client window.  */

MRESULT
Create (HWND hwnd)
{
  DEVOPENSTRUC dop;
  BITMAPINFOHEADER2 bmih;

  /* Fix the display size.  */

  ptsByteAlign.x = WinQuerySysValue (HWND_DESKTOP, SV_CXBYTEALIGN);
  if (ptsByteAlign.x == 0)
    ptsByteAlign.x = 8;

  ptsByteAlign.y = WinQuerySysValue (HWND_DESKTOP, SV_CYBYTEALIGN);
  if (ptsByteAlign.y == 0)
    ptsByteAlign.y = 8;

  sizlGeometry.cx = RoundDown (ulWidth, ptsByteAlign.x);
  sizlGeometry.cy = RoundDown (ulHeight, ptsByteAlign.y);

  /* Setup the color map.  */

  alColorMap[MF_WHITE] = CLR_BACKGROUND;
  alColorMap[MF_BLACK] = CLR_NEUTRAL;

  /* Open the device context for the screen and create a normal
     presentation space for painting.  */

  hdcScreen = WinOpenWindowDC (hwnd);
  if (!hdcScreen)
    Bye (WIN, habMain, "WinOpenWindowDC", LINE (2));

  hpsScreen = GpiCreatePS (habMain, hdcScreen, &sizlGeometry,
    PU_PELS | GPIF_LONG | GPIT_NORMAL | GPIA_ASSOC);
  if (!hpsScreen)
    Bye (WIN, habMain, "GpiCreatePS", LINE (3));

  /* Open the memory device context and set the drawing mode to retained
     graphics.  */

  dop.pszLogAddress = NULL;
  dop.pszDriverName = "DISPLAY";

  hdcMemory = DevOpenDC (habMain, OD_MEMORY, "*",
    2, (PDEVOPENDATA) &dop, NULLHANDLE);
  if (!hdcMemory)
    Bye (WIN, habMain, "DevOpenDC", LINE (3));

  hpsMemory = GpiCreatePS (habMain, hdcMemory, &sizlGeometry,
    PU_PELS | GPIF_LONG | GPIT_NORMAL | GPIA_ASSOC);
  if (!hpsMemory)
    Bye (WIN, habMain, "GpiCreatePS", LINE (3));

  fSuccess = GpiSetDrawingMode (hpsMemory, DM_RETAIN);
  if (!fSuccess)
    Bye (WIN, habMain, "GpiSetDrawingMode", LINE (2));

  /* Create a bitmap as big as the screen and make it the current bitmap for
     the memory presentation space.  */

  memset (&bmih, 0, sizeof (BITMAPINFOHEADER2));

  bmih.cbFix = sizeof (BITMAPINFOHEADER2);
  bmih.cx = sizlGeometry.cx;
  bmih.cy = sizlGeometry.cy;
  bmih.cPlanes = 1;
  bmih.cBitCount = 1;

  hbmImage = GpiCreateBitmap (hpsMemory, &bmih, 0, NULL, NULL);
  if (!hbmImage)
    Bye (WIN, habMain, "GpiCreateBitmap", LINE (2));

  GpiSetBitmap (hpsMemory, hbmImage);

  /* Make sure that nobody writes into core during painting.  `TRUE' means
     `owned', here.  */

  apiretError = DosCreateMutexSem (NULL, &hmtxImage, 0, TRUE);
  if (apiretError)
    Bye (DOS, "DosCreateMutexSem", LINE (2));

  fSuccess = GpiErase (hpsMemory);
  if (!fSuccess)
    Bye (WIN, habMain, "GpiErase", LINE (2));

  apiretError = DosReleaseMutexSem (hmtxImage);
  if (apiretError)
    Bye (DOS, "DosReleaseMutexSem", LINE (2));

  /* Now replace "hwnd" by its parent.  This is the same window as
     "hwndMain" but WinCreateStdWindow() hasn't returned yet.  */

  hwnd = WinQueryWindow (hwnd, QW_PARENT);
  if (!hwnd)
    Bye (WIN, habMain, "WinQueryWindow", LINE (2));

  /* See if the server is present in the window list and disable switching
     if not (we cannot use getppid() because this is a PM session and our
     parent is the window system itself).  */

  hwndMenu = WinWindowFromID (hwnd, FID_MENU);
  if (!hwndMenu)
    Bye (WIN, habMain, "WinWindowFromID", LINE (2));

  hswitchMetafont = WinQuerySwitchHandle (0, pidMetafont);
  if (!hswitchMetafont)
    WinSendMsg (hwndMenu, MM_SETITEMATTR,
      MPFROM2SHORT (IDM_FILE_SUSPEND, TRUE),
      MPFROM2SHORT (MIA_DISABLED, MIA_DISABLED));

  /* Get the scroll bar handles of the frame window.  We do not have to
     initialize them now because a "WM_SIZE" message will be posted to the
     client window after this (see Resize() below).  */

  hwndHorizScroll = WinWindowFromID (hwnd, FID_HORZSCROLL);
  if (!hwndHorizScroll)
    Bye (WIN, habMain, "WinWindowFromID", LINE (2));

  hwndVertScroll = WinWindowFromID (hwnd, FID_VERTSCROLL);
  if (!hwndVertScroll)
    Bye (WIN, habMain, "WinWindowFromID", LINE (2));

  /* Show the upper left corner of the presentation space in accordance to
     the position of the scroll bars. */

  ptsScrollOld.x = 0;
  ptsScrollOld.y = 0;

  ptsScrollPos.x = 0;
  ptsScrollPos.y = 0;

  return (MRESULT) 0;
}


/* Adjust the scroll bars to the current value of "ptsScrollPos".  We
   must not redraw the client window because we have the class style
   "CS_SIZEREDRAW" set.  */

MRESULT
Resize (HWND hwnd, MPARAM mp1, MPARAM mp2)
{
  /* Adjust the size of the scrollable window.  */

  ptsScrollSize.x = SHORT1FROMMP (mp2);
  ptsScrollSize.y = SHORT2FROMMP (mp2);

  /* Compute the scroll steps.  */

  ptsScrollPage.x = RoundDown (ptsScrollSize.x >> 1, ptsByteAlign.x);
  ptsScrollPage.y = RoundDown (ptsScrollSize.y >> 1, ptsByteAlign.y);
  ptsScrollLine.x = RoundDown (ptsScrollSize.x >> 5, ptsByteAlign.x);
  ptsScrollLine.y = RoundDown (ptsScrollSize.y >> 5, ptsByteAlign.y);

  /* Set the position, range and size for both scroll bars.  */

  fSuccess = (BOOL) WinSendMsg (hwndHorizScroll, SBM_SETSCROLLBAR,
     MPFROMLONG (ptsScrollPos.x), MPFROM2SHORT (0, (SHORT) sizlGeometry.cx));
  if (!fSuccess)
    Bye (WIN, habMain, "WindSendMsg", LINE (3));

  fSuccess = (BOOL) WinSendMsg (hwndVertScroll, SBM_SETSCROLLBAR,
    MPFROMLONG (ptsScrollPos.y), MPFROM2SHORT (0, (SHORT) sizlGeometry.cy));
  if (!fSuccess)
    Bye (WIN, habMain, "WindSendMsg", LINE (3));

  fSuccess = (BOOL) WinSendMsg (hwndHorizScroll, SBM_SETTHUMBSIZE,
    MPFROM2SHORT (ptsScrollSize.x, sizlGeometry.cx), MPVOID);
  if (!fSuccess)
    Bye (WIN, habMain, "WinSendMsg", LINE (3));

  fSuccess = (BOOL) WinSendMsg (hwndVertScroll, SBM_SETTHUMBSIZE,
    MPFROM2SHORT (ptsScrollSize.y, sizlGeometry.cy), MPVOID);
  if (!fSuccess)
    Bye (WIN, habMain, "WinSendMsg", LINE (3));

  return (MRESULT) 0;
}


/* Copy the `visible' part of the image to the screen.  */

MRESULT
Paint (HWND hwnd)
{
  POINTL aptlCoord[3];

  /* The client window is the target rectangle.  */

  aptlCoord[0].x = 0;
  aptlCoord[0].y = 0;

  aptlCoord[1].x = ptsScrollSize.x;
  aptlCoord[1].y = ptsScrollSize.y;

  /* Is this save arithmetic for the lower left corner coordinates of the
     image?  This is an unusual way of scrolling but absolutely simple.  */

  aptlCoord[2].x = (sizlGeometry.cx - ptsScrollSize.x)
    * ptsScrollPos.x / sizlGeometry.cx;
  aptlCoord[2].y = (sizlGeometry.cy - ptsScrollSize.y)
    * (sizlGeometry.cy - ptsScrollPos.y) / sizlGeometry.cy;

  /* Do not paint during bitmap updates.  */

  apiretError = DosRequestMutexSem (hmtxImage, SEM_INDEFINITE_WAIT);
  if (apiretError)
    Bye (DOS, "DosRequestMutexSem", LINE (2));

  /* OK, display the new view point.  */

  fSuccess = WinBeginPaint (hwnd, hpsScreen, NULL);
  if (!fSuccess)
    Bye (WIN, habMain, "WinBeginPaint", LINE (2));

  fSuccess = GpiBitBlt (hpsScreen, hpsMemory,
    3, aptlCoord, ROP_NOTSRCCOPY, BBO_IGNORE);
  if (!fSuccess)
    Bye (WIN, habTalk, "GpiBitBlt", LINE (3));

  fSuccess = WinEndPaint (hpsScreen);
  if (!fSuccess)
    Bye (WIN, habMain, "WinEndPaint", LINE (2));

  /* We are done.  */

  apiretError = DosReleaseMutexSem (hmtxImage);
  if (apiretError)
    Bye (DOS, "DosReleaseMutexSem", LINE (2));

  return (MRESULT) 0;
}


/* Pick the commands of interests from the message queue. */

MRESULT
Command (HWND hwnd, MPARAM mp1, MPARAM mp2)
{
  switch (SHORT1FROMMP (mp1))
    {
    case IDM_FILE_SUSPEND:

      /* Ignore errors, here.  */

      WinSwitchToProgram (hswitchMetafont);

      return (MRESULT) 0;

    case IDM_FILE_EXIT:

      /* kill (getpid (), SIGTERM);  */

      fSuccess = WinPostMsg (hwndMain, WM_CLOSE, MPVOID, MPVOID);
      if (!fSuccess)
	Bye (WIN, habMain, "WinPostMsg", LINE (2));

      return (MRESULT) 0;
    }

  return WinDefWindowProc (hwnd, WM_COMMAND, mp1, mp2);
}


/* Scroll the window contents horizontally and vertically.  */

LONG
SliderPos (MPARAM mp2, ULONG ulMessage)
{
  QMSG qmsgPeek;			/* Message structure. */

  while (WinPeekMsg (habMain, &qmsgPeek, 0, ulMessage, ulMessage, PM_NOREMOVE))
    {
      if (!(SHORT2FROMMP (qmsgPeek.mp2) == SB_SLIDERTRACK ||
	    SHORT2FROMMP (qmsgPeek.mp2) == SB_SLIDERPOSITION))
	break;

      if (!WinPeekMsg (habMain, &qmsgPeek, 0, ulMessage, ulMessage, PM_REMOVE))
	Bye (WIN, habMain, "WinPeekMsg", LINE (1));

      mp2 = qmsgPeek.mp2;
    }

  return SHORT1FROMMP (mp2);
}


MRESULT
HorizScroll (HWND hwnd, MPARAM mp1, MPARAM mp2)
{
  LONG lDiff;				/* Scroll that many items. */

  switch (SHORT2FROMMP (mp2))
    {
    case SB_LINELEFT:
      ptsScrollPos.x -= ptsScrollLine.x;
      break;
    case SB_LINERIGHT:
      ptsScrollPos.x += ptsScrollLine.x;
      break;
    case SB_PAGELEFT:
      ptsScrollPos.x -= ptsScrollPage.x;
      break;
    case SB_PAGERIGHT:
      ptsScrollPos.x += ptsScrollPage.x;
      break;
    case SB_SLIDERTRACK:
    case SB_SLIDERPOSITION:
      ptsScrollPos.x = RoundDown (SliderPos (mp2, WM_HSCROLL), ptsByteAlign.x);
      break;
    }

  if (ptsScrollPos.x < 0)
    ptsScrollPos.x = 0;
  if (ptsScrollPos.x > sizlGeometry.cx)
    ptsScrollPos.x = sizlGeometry.cx;

  lDiff = ptsScrollPos.x - ptsScrollOld.x;
  if (lDiff)
    {
      fSuccess = WinInvalidateRect (hwndClient, NULL, FALSE);
      if (!fSuccess)
	Bye (WIN, habMain, "WinInvalidateRect", LINE (2));

      fSuccess = (BOOL) WinSendMsg (hwndHorizScroll,
        SBM_SETPOS, MPFROM2SHORT (ptsScrollPos.x, 0), MPVOID);
      if (!fSuccess)
	Bye (WIN, habMain, "WinSendMsg", LINE (3));

      ptsScrollOld.x = ptsScrollPos.x;
    }

  return (MRESULT) 0;
}


MRESULT
VertScroll (HWND hwnd, MPARAM mp1, MPARAM mp2)
{
  SHORT lDiff;

  switch (SHORT2FROMMP (mp2))
    {
    case SB_LINEUP:
      ptsScrollPos.y -= ptsScrollLine.y;
      break;
    case SB_LINEDOWN:
      ptsScrollPos.y += ptsScrollLine.y;
      break;
    case SB_PAGEUP:
      ptsScrollPos.y -= ptsScrollPage.y;
      break;
    case SB_PAGEDOWN:
      ptsScrollPos.y += ptsScrollPage.y;
      break;
    case SB_SLIDERTRACK:
    case SB_SLIDERPOSITION:
      ptsScrollPos.y = RoundDown (SliderPos (mp2, WM_VSCROLL), ptsByteAlign.y);
      break;
    }

  if (ptsScrollPos.y < 0)
    ptsScrollPos.y = 0;
  if (ptsScrollPos.y > sizlGeometry.cy)
    ptsScrollPos.y = sizlGeometry.cy;

  lDiff = ptsScrollPos.y - ptsScrollOld.y;
  if (lDiff)
    {
      fSuccess = WinInvalidateRect (hwndClient, NULL, FALSE);
      if (!fSuccess)
	Bye (WIN, habMain, "WinInvalidateRect", LINE (2));

      fSuccess = (BOOL) WinSendMsg (hwndVertScroll,
        SBM_SETPOS, MPFROM2SHORT (ptsScrollPos.y, 0), MPVOID);
      if (!fSuccess)
	Bye (WIN, habMain, "WinSendMsg", LINE (3));

      ptsScrollOld.y = ptsScrollPos.y;
    }

  return (MRESULT) 0;
}


/* Map keyboard input into window messages.  */

MRESULT
KeyEvent (HWND hwnd, MPARAM mp1, MPARAM mp2)
{
  ULONG ulFlags;			/* Keyboard flags. */
  ULONG ulChar;				/* Character code. */
  ULONG ulKey;				/* Virtual key code. */

  ulFlags = (ULONG) SHORT1FROMMP (mp1);
  ulChar = (ULONG) SHORT1FROMMP (mp2);
  ulKey = (ULONG) SHORT2FROMMP (mp2);

  if (ulFlags & KC_VIRTUALKEY)
    {
      switch (ulKey)
	{
	case VK_LEFT:
	case VK_RIGHT:
	  return WinSendMsg (hwndHorizScroll, WM_CHAR, mp1, mp2);
	case VK_UP:
	case VK_DOWN:
	case VK_PAGEUP:
	case VK_PAGEDOWN:
	  return WinSendMsg (hwndVertScroll, WM_CHAR, mp1, mp2);
	}
    }

  return WinDefWindowProc (hwnd, WM_CHAR, mp1, mp2);
}


/* Define two functions for rounding a number to a multiple of another. */

LONG
RoundDown (LONG lArg, LONG lMod)
{
  if (lArg < 0)
    {
      lArg -= lMod - 1;
      lArg += lArg % lMod;
    }
  else
    lArg -= lArg % lMod;

  return lArg;
}

LONG
RoundUp (LONG lArg, LONG lMod)
{
  if (lArg < 0)
    lArg -= lArg % lMod;
  else
    {
      lArg += lMod - 1;
      lArg -= lArg % lMod;
    }

  return lArg;
}
